extensions [ nw ]

globals [ culture-breastfeeding-maximum zika-monthly-recovery-chance zika-births ]

turtles-own [ age infected? contracepting? menopause? fertile? pregnant? months-pregnant months-postpartum breastfeeding ideal-family-size parity fear foolhardy bore-zika-baby ]

to setup
  clear-all
  set culture-breastfeeding-maximum 7
  set zika-monthly-recovery-chance 10
  set zika-births 0

  nw:generate-preferential-attachment turtles links n-agents [
;; initialise agent counters and Boolean variables
    set parity 0
    set months-pregnant 0
    set months-postpartum 0
    set pregnant? FALSE
    set menopause? FALSE
    set fertile? TRUE
    set contracepting? FALSE
    set infected? FALSE
    set bore-zika-baby 0
    set fear 0
    set foolhardy random 30
;; age of agent in years
    set age 21 + random 10
;; how many offspring agent wants
    set ideal-family-size random-poisson culture-ideal-family-size
    set shape "person"
    set color white
    setxy random-xcor random-ycor
    if show-labels? [
      set label (word age " " parity " " ideal-family-size " " months-pregnant " " months-postpartum)
    ]
  ]
  repeat 300 [ layout-spring turtles links 0.2 7 1.3 ]

  reset-ticks
end

to go

  ask turtles [

;; pregnant women only
    ifelse pregnant? [
      set months-pregnant months-pregnant + 1
      if months-pregnant > 9 [
        birth
      ]
    ]

;; non-pregnant women
    [
      if months-postpartum > 0 [
        set fertile? FALSE
        set months-postpartum months-postpartum - 1
      ]
      ifelse menopause? [
        set fertile? FALSE
      ] [
        decide-contracept
      ]
      if fertile? [
        if random 100 > 90 [
          conceive
        ]
      ]
    ]

;; all women
    if infected? [
      if random 100 < zika-monthly-recovery-chance [
        set infected? FALSE
        set pcolor black
      ]
    ]
    if random 100 < zika-prevalence * 10 [
      set infected? TRUE
      set pcolor brown
    ]
    calculate-fear

    colour-status
    if show-labels? [
      set label (word age " " parity " " ideal-family-size " " months-pregnant " " months-postpartum)
    ]
  ] ;; end of main ask turtles

  if all? turtles [ menopause? = TRUE OR parity = ideal-family-size ] [
    stop
  ]
  age-increment
  tick
end

to colour-status
    ifelse pregnant? [
       set color red
    ]
    [ if months-postpartum = 0 [
         set color white
      ]
    ]
    ifelse months-postpartum > 0 [
       set color blue
    ]
;; not postpartum
    [ if contracepting? [
         set color green
      ]
    ]
    if menopause? = TRUE [
       set color grey
    ]
end

to conceive
  set pregnant? TRUE
  set months-pregnant 1
end

to birth
  if infected? [
    zika-birth
  ]
;; add 1 to number of offspring ever birthed by this agent
  set parity parity + 1
  set pregnant? FALSE
  set months-pregnant 0
  set breastfeeding random culture-breastfeeding-maximum
;; number of months before agent regains possibility of conceiving
  set months-postpartum 1 + breastfeeding
end

;; every 12th month, age of all agents is incremented by 1 year
to age-increment
  if ticks mod 12 = 0 [
    ask turtles [
      set age age + 1
;; annual check for menopause starting
      if menopause? = FALSE [
        if age > 40 [
          if random 100 < 10 [
            set menopause? TRUE
          ]
        ]
        if age > 50 [
          if random 100 < 50 [
            set menopause? TRUE
          ]
        ]
        if age > 55 [
          set menopause? TRUE
        ]
      ]
    ]
  ]
end

to calculate-fear
;; initialise local variables
  let background 0
  let impact-of-self-infected 0
  let impact-of-self-bearing-zika-baby 0
  let n-friends-infected 0
  let impact-of-friend-infected 0
  let n-friends-had-zika-baby 0
  let impact-of-friend-having-zika-baby 0
  let n-others-had-zika-baby 0
  let impact-of-others-having-zika-baby 0

  set background ( zika-prevalence * 10 )

;; self is infected
  if infected? [ set impact-of-self-infected 30 ]

  if bore-zika-baby > 0 [ set impact-of-self-bearing-zika-baby 80 ]

;; friend is infected
  set n-friends-infected ( count link-neighbors with [ infected? ] )
  set impact-of-friend-infected ( n-friends-infected * 10 )

;; friend bore Zika baby
  set n-friends-had-zika-baby round ( sqrt count link-neighbors with [ bore-zika-baby > 0 ] )
  set impact-of-friend-having-zika-baby ( n-friends-had-zika-baby * 40 )

;; other bore Zika baby
  set n-others-had-zika-baby ( zika-births - n-friends-had-zika-baby )
  set impact-of-others-having-zika-baby ( round ( sqrt n-others-had-zika-baby * 10 ) )

  set fear background + impact-of-self-infected + impact-of-self-bearing-zika-baby + impact-of-friend-infected + impact-of-friend-having-zika-baby + impact-of-others-having-zika-baby - foolhardy

;; debugging the components of fear
;;  show ( word background " FI:" impact-of-friend-infected " FB:" impact-of-friend-having-zika-baby " O:" impact-of-others-having-zika-baby " R:-" foolhardy )
end

to decide-contracept
  set contracepting? FALSE
  set fertile? TRUE
  let contracept? FALSE

;; family incomplete so maybe try for conception if not too risky
  ifelse parity < ideal-family-size [

    if random 100 < fear [
      set contracept? TRUE
    ]
  ]
;; family complete
  [
    set contracept? TRUE
  ]
  if contracept? [
    set contracepting? TRUE
    set fertile? FALSE
  ]
end

to zika-birth
;; record that agent had a Zika-affected baby
  set bore-zika-baby bore-zika-baby + 1
;; increment cohort total
  set zika-births zika-births + 1
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
1200
541
24
12
20.0
1
12
1
1
1
0
1
1
1
-24
24
-12
12
1
1
1
ticks
30.0

SLIDER
10
10
191
43
n-agents
n-agents
10
50
30
1
1
NIL
HORIZONTAL

BUTTON
10
206
65
239
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
73
206
128
239
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
11
264
98
309
All births
sum [ parity ] of turtles
17
1
11

SLIDER
11
101
191
134
zika-prevalence
zika-prevalence
0
1
0.5
0.1
1
NIL
HORIZONTAL

SLIDER
10
55
191
88
culture-ideal-family-size
culture-ideal-family-size
0
7
2
1
1
NIL
HORIZONTAL

BUTTON
136
207
191
240
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
107
264
190
309
Zika births
zika-births
17
1
11

MONITOR
11
319
191
364
Sum of "ideal family size"
sum [ ideal-family-size ] of turtles
17
1
11

SWITCH
12
146
150
179
show-labels?
show-labels?
0
1
-1000

PLOT
209
558
685
708
Virus-affected births
Months
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"zika" 1.0 0 -2674135 true "" "plot zika-births"

PLOT
701
558
1201
708
Total Births
Months
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"sumbirths" 1.0 0 -16777216 true "" "plot sum [ parity ] of turtles"

MONITOR
11
388
88
433
parity = 0
count turtles with [ parity = 0 ]
17
1
11

MONITOR
11
436
88
481
parity =1
count turtles with [ parity = 1 ]
17
1
11

MONITOR
12
484
88
529
parity = 2
count turtles with [ parity = 2 ]
17
1
11

MONITOR
13
532
87
577
parity = 3
count turtles with [ parity = 3 ]
17
1
11

MONITOR
13
628
134
673
parity 5 or higher
count turtles with [ parity > 4 ]
17
1
11

MONITOR
13
580
87
625
parity = 4
count turtles with [ parity = 4 ]
17
1
11

@#$#@#$#@
## WHAT IS IT?

This models reproductive decisions during life history in a context of fear of malformed babies caused by an endemic virus. For topicality the word Zika is used but this does not attempt realistic simulation of the epidemiology of that virus.

Each agent represents one woman.

The agent's reproductive condition is colour-coded:
white = susceptible to starting a pregnancy
red = pregnant
blue = postpartum infecundability (immediately after giving birth a new conception is unlikely)
green = contracepting
grey = after the menopause

For simplicity, the effect of breastfeeding on lowering the probability of conception is treated as temporary infertility added to the postpartum infecundability time.

All agents in the model cohort are potentially fertile, in a marital relationship, and susceptible to pregancy in any month until menopause unless affected by postpartum infecundability or using contraception.

The only environmental feature is an endemic virus. The agent's patch colour changes to brown when the agent is infected by the virus, and it returns to the background colour if she recovers.

Labels on each agent show five variables in this sequence left to right: age, parity, ideal-family-size, months-pregnant, months-postpartum. Comparing parity (number of offspring) and ideal-family-size lets you see if the agent has yet completed her family at the number she wished.

## HOW IT WORKS

SETUP creates agents representing women, the number determined by variable n-agents, arranged in a preferential-attachment social network. For simplicity the cohort of women (perhaps contemporaries from one college) stays constant as they age, and no one dies.

GO simulates one month of reproductive life history. For each woman, if pregnant the months are counted until birth. If fertile but not pregnant a decision is made whether to use contraception in that month. Then viral status is updated with a chance of becoming infected, and also a chance for an infected person to recover. Each agent calculates her fear of a deformed birth, based on awareness of infection and deformed births affecting three categories of people - self, social neighbours, and other people - with personal experience weighted highest and non-neighbours' experiences weighted lowest. Then agents are coloured to indicate condition and labels are rewritten. After a test to stop the simulation, each agents has one month added to her age.

The test stops the simulation if everyone has either completed her family (had as many births as she wants) or become infertile by reaching menopause.

Agent variables

ideal-family-size (a target number of children)
fear (level of perceived risk of a virus-affected birth)
foolhardy (tendency to ignore risk

Counters
age (in years, incremented at every 12th month tick)
months-pregnant (number of months since conception)
postpartum (number of months of infecundability after birth)
parity (number of times a woman has given birth)
bore-zika-baby (virus-affected babies born to this woman)

State
contracepting? (using contraception this month)
menopause? (menopause has happened)
(when either is TRUE the woman will not become pregnant)

Derived variable
fertile? (short for: [ postpartum > 0 OR contracepting? OR menopause? ] )

Environment variables
culture-breastfeeding-maximum (the number of months beyond which breastfeeding is normally stopped)
zika-monthly-recovery-chance (probability % of removing infection from woman agent)

Counters
zika-births (the number of virus-affected babies among the cohort)

Agent actions

The decision to contracept first checks if the number of offspring has reached the agent's ideal family size. If so then contraception will be used until menopause.

The level of fear of virus effects is shaped by knowledge of infections and birth effects among close friends and (with a smaller degree of weighting) the wider cohort. If fear exceeds a threshold the agent will defensively contracept.

## HOW TO USE IT

Before SETUP:
n-agents slider sets the number of women.
zika-prevalence slider sets the rate of viral infection.
culture-ideal-family-size slider sets a norm, from which using a Poisson distribution each woman has an individualised target.

Try single GO steps initially to see pregnancies beginning, infections, contraceptive use decisions. Also inspect the labels. The numbers immediately left and right of a turtle are the counters for pregnancy and postpartum infecundability. The other numbers, starting leftmost, are age, ideal family size, and parity (number of offspring).

Since the cohort of women is fixed, and the simulation only ends when no agents want to or can give birth, the latter half of most runs will be mostly uneventful and I suggest increasing the Netlogo speed to maximum after around 120 months.

Inputs
culture-ideal-family-size = typical aspiration regarding number of offspring.
zika-prevalence =  the rate at which people become infected by virus.

Outputs
Any of the counters can be treated as outputs, but those plotted are total numbers of births and virus-affected births within this cohort of women.

## THINGS TO NOTICE

Due to the Poisson distribution some women may have a zero ideal family size (childfree) and any such agents will start contracepting immediately.

## THINGS TO TRY

Try adjusting the culture-ideal-family-size while infection chance is zero and see the outcomes.

Try adjusting the infection prevalence and see what effect it has.

## EXTENDING THE MODEL

Immunity to the virus could be added. Currently a recovered person is susceptible to re-infection (or relapse) at the same rate as initially.

Age could affect the probability of conception.

The ages could be recorded at which each agent has first birth, second birth, etc. That would allow calculation of spacing (time intervals between births). Also could record the age at which menopause occurs.

The percentage of months spent in each condition (e.g. contracepting, pregnant, etc.) could be recorded for each agent.

The effect of breastfeeding is not guaranteed contraception but a reduced probability of conception, and over time that slowly increases. If realistic gradation were in place then the slider maximum breastfeeding duration could be raised to three years as was normal in some pre-modern cultures.

The minimum age for starting reproduction (21 years in model) could be adjusted.

## NETLOGO FEATURES

The social network uses Netlogo's preferential attachment.

Poisson distribution is used for each agent's ideal-family-size.

## Credits

Created using NetLogo version 5.3.1

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="60"/>
    <metric>sum [ parity ] of turtles</metric>
    <enumeratedValueSet variable="n-agents">
      <value value="20"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
